# Simulate coalescent process with heterochronous sampling times
sqWvFixFn <- function(tshift, nsegs, numTraj, nwave, wave, Nmax, idTraj, numRuns, nSamps, samptimes){
  # Assumptions and modifications
  # - error in sqwave traj fixed
  # - single trajectory type considered
  # - reorganises data so in a single folder
  # - assumes fixed number of samples (elements of nSamps)
  # - model selection over square waves of 2^p segments
  
  # Function to write simple csv files to correct path
  tableWrite <- function(val, name, pathname) {
    # Add path to name
    str0 <- paste(c(pathname, name), collapse = "")
    # Write table
    write.table(val, str0, row.names=FALSE, col.names=FALSE, sep=",")
  }
  
  # Main code for heterochronous simulations ----------------------------------------------------------

  # Create folder for traj specific results
  trajName = paste(c('sqwaves_', idTraj), collapse = '')
  dir.create(file.path(this.dir, trajName))
  pathf = paste(c(this.dir, '/', trajName, '/'), collapse = "")
  # Fix trajectories (replicates)
  idTraj = rep(idTraj, numTraj)
    
  for (j in 1:numRuns) {
    # Number of samples introduced at each time
    nsamps = nSamps[j]
    samps = nsamps*rep(1, nsegs)
    # Coalescent events in each trajectory
    nc = rep(0, numTraj)
    
    # Define square wave population size
    sqwave_traj <- function (t, wav = wave[idTraj[j],], N = Nmax, tseg = samptimes) 
    {
      popsize = rep(N, length(t))
      # Changing population based on wave
      for(i in 1:nsegs-1){
        popsize[t < tseg[i+1] & t >= tseg[i]] = wav[i]*N
      }
      # Beyond last wave segment hold last popsize
      popsize[t >= tseg[nsegs]] = wav[nsegs]*N
      return(popsize)
    }
    traj = sqwave_traj
    
    # Store all coalescent data
    coalT = matrix(list(), 1, numTraj)
    coalL = matrix(list(), 1, numTraj)
    
    for(i in 1:numTraj){
      # Simulate genealogy and get all times
      gene = coalsim(samp_times = samptimes, n_sampled = samps, traj = traj, lower_bound = 10, method = "thin")
      coal_times = gene$coal_times 
      coalT[[i]] = coal_times
      coalLin = gene$lineages
      coalL[[i]] = coalLin
      
      # TMRCA and no. coalescent events
      tmax = max(coal_times)
      nc[i] = length(coal_times)
    }
    # Unlist so can write numerically
    coalL = matrix(unlist(coalL), ncol = numTraj, nrow = unique(nc))
    coalT = matrix(unlist(coalT), ncol = numTraj, nrow = unique(nc))
    # These change with runs and replicates
    tableWrite(coalT, paste(c('coalT', j, '.csv'), collapse = ''), pathf)
    tableWrite(coalL, paste(c('coalL', j, '.csv'), collapse = ''), pathf)
    
    # No. samples, coalescences and times
    tableWrite(samptimes, 'samptimes.csv', pathf)
    tableWrite(nSamps, 'sampRuns.csv', pathf)
    
    # No. runs and replicates
    tableWrite(numRuns, 'numRuns.csv', pathf)
    tableWrite(numTraj, 'numTraj.csv', pathf)
    
    # Data on square wave models
    tableWrite(wave, 'wave.csv', pathf)
    tableWrite(idTraj, 'idTraj.csv', pathf)
    tableWrite(c(Nmax, frac*Nmax), 'Nrange.csv', pathf)
  }
}




